/** @file   npccharacter.h
 * @brief   Declaration of NpcCharacter - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_NPCCHARACTER_H
#define H_WWW_NPCCHARACTER_H

#include "gameobject.h"
#include "weapon.h"

// Forward declarations
namespace eng2d {
  class AnimItem;
};

namespace WeWantWar {

/** @class  NpcCharacter
 * @brief   These are the special characters that could help the player.
 * @author  Tomi Lamminsaari
 *
 * All the NPC - characters are friendly to the player.
 */
class NpcCharacter : public GameObject
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** Index of the counter we use when calculating the reloading delays. */
  static const int RELOAD_COUNTER_INDEX = 0;
  /** Index of the controlpoint that marks the nose of the gun */
  static const int GUN_NOSE_CTRLPOINT = 0;

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	NpcCharacter();


	/** Destructor
   */
	virtual ~NpcCharacter();

private:

	/** Copy constructor.
   * @param     rO                Reference to another NpcCharacter
   */
  NpcCharacter( const NpcCharacter& rO );

	/** Assignment operator
   * @param     rO                Reference to another NpcCharacter
   * @return    Reference to us.
   */
  NpcCharacter& operator = ( const NpcCharacter& rO );

public:

  ///
  /// Methods
  /// =======

  /** Redraws the Character
   * @param     pQ                Pointer to redrawing queue.
   */
  virtual void redraw( RedrawQueue* pQ );

  /** Makes sound. To make different sounds, update the sample numbers
   * in m_soundIndices - vector.
   * @param     id                ID of the sound to make
   */
  virtual void makeSound( GameObject::SoundID id ) const;

  /** Handles the bullethits. Default implements ignores the bullets shot
   * by player.
   * @param     pB                Pointer to bullet that hit us
   * @return    true if we accept the bullet hit or false if the bullet
   *            goes right through.
   */
  virtual bool hitByBullet( Bullet* pB );

  /** Kills this object
   */
  virtual void kill();
  
  /** Sets the weapon
   * @param     w                 New weapon
   */
  virtual void setWeapon( Weapon w );
  

  ///
  /// Getter methods
  /// ==============
  
  /** Tells if we're reloading our weapon.
   * @return    true if we're reloading.
   */
  virtual bool reloading() const;
  
  /** Returns the weapon this object has.
   * @return    The weapon
   */
  virtual Weapon getWeapon() const;


protected:

  /** Sets the requested animation unless it's already running. The
   * m_pAnimVector - member must have valid pointer to animation vector. If
   * it's not valid, this method will crash the program.
   * @param     animID            ID code of the animation. See GameAnims - class.
   */
  void changeAnimation( int animID );
  
  /** Shoots. This method checks the weapon we're carrying and shoots with it.
   */
  void handleShooting();
  
  /** Updates the energybar buffer
   */
  void updateEnergybar();
  

  ///
  /// Members
  /// =======

  /** Pointer to Animation Combine that contains the animations of this
   * character.
   */
  const eng2d::AnimContainer* m_animationCombine;

  /** The weapon this object has. */
  Weapon  m_weapon;

  /** This vector holds the sample numbers the makeSound() - method uses
   * when it should produce sound. Use the SoundID as index.
   */
  std::vector<int>  m_soundIndices;
  
  /** Pointer to bitmap that we need when showing the energy bar */
  BITMAP* iEnergybar;
  
  /** A counter that tells the next time we should update the energy bar */
  int     iEnergybarUpdate;
};


};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: npccharacter.h,v $
 * Revision 1.3  2006/02/24 12:25:15  lamminsa
 * no message
 *
 * Revision 1.2  2006/02/08 22:03:28  lamminsa
 * New animation and graphic system was implemented.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:06+02  lamminsa
 * Initial revision
 *
 */
 
